/** @file   PlayerCar.h
 * @brief   Declarations of PlayerCar - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:42 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_PLAYERCAR_H
#define H_WWW_PLAYERCAR_H

#include "polygonobject.h"
#include "MCapturable.h"

//
// Forward declarations


namespace WeWantWar {


/** @class  PlayerCar
 * @brief   Represents the car the player uses.
 * @author  Tomi Lamminsaari
 
 */
class PlayerCar : public PolygonObject, public MCapturable
{
public:
  ///
  /// Static data, datatypes and constants
  /// ====================================

  /** Index of the counter we use for reloading delays */
  static const int KReloadingCounterIndex = 0;
  
  /** Index of the counter that calculates the delays how soon the
   * player can exit this vehicle once he has captured it.
   */
  static const int KNoExitCounterIndex = 1;
  
  /** Index of the control point that tells the position of the
   * machine gun.
   */
  static const int KMachinegunCtrlPointIndex = 0;

  ///
  /// Static methods
  /// ==============



  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructor.
   * @param   aRouteName        Name of the route this car should drive.
   */
  PlayerCar( const std::string& aRouteName );

  /** Destructor.
   */
  virtual ~PlayerCar();

  /** Copy constructor.
   * @param   aAnother            Reference to another PlayerCar
   */
  PlayerCar( const PlayerCar& aAnother );

  /** An assignment operator.
   * @param   aAnother            Reference to another PlayerCar
   * @return  Reference to this object.
   */
  PlayerCar& operator = ( const PlayerCar& aAnother );


  ///
  /// Inherited methods
  /// =================

  /** Updates this car.
   */
  virtual void update();
  
  /** Adds the graphics of this object to redraw queue.
   * @param   aQueue              Pointer to redraw queue
   */
  virtual void redraw( RedrawQueue* aQueue );
  
  /** Makes sound.
   * @param   aSoundId            Id of the sound we must play.
   */
  virtual void makeSound( GameObject::SoundID aSoundId ) const;
  
  /** Destroys this object.
   */
  virtual void kill();
  
  /** Handles the bullet hits
   * @param   aBullet             Pointer to bullet that hit us.
   * @return  <code>true</code> if accepted the bullet hit.
   */
  virtual bool hitByBullet( Bullet* aBullet );
  
  /** From MCapturable. This object gets captured by the object.
   * @param   aMaster             Pointer to object who captured us.
   */
  virtual void capture( GameObject* aMaster );
  
  /** From MCapturable. Resurrects this object.
   */
  virtual void resurrect();
  

  ///
  /// New public methods
  /// ==================



  ///
  /// New getter methods
  /// ==================

  /** From MCapturable. Tells if this object can be captured.
   * @return  true if we can be captured.
   */
  virtual bool canBeCaptured() const;
  
  /** From MCapturable. Returns the pointer to our master.
   * @return  Pointer to our master object or nullpointer.
   */
  virtual GameObject* getMaster() const;

  /** Are we in reloading state.
   * @return  Always false
   */
  virtual bool reloading() const;
  
  /** Returns the type of this object.
   * @return  ObjectID::TYPE_PLAYERCAR
   */
  virtual ObjectID::Type objectType() const;
  


protected:

  ///
  /// Protected interface
  /// ===================
  
  /** Updates the captured car
   */
  void updateCaptured();

  /** Rotates the machinegun.
   * @param   aAngleChange        Amount of angle change.
   */
  void rotateMachineGun( int aAngleChange );
  
  /** Shoots
   */
  void shoot();


protected:

  ///
  /// Member data
  /// ===========

  /** Pointer to our master object. */
  GameObject* iMaster;
  
  /** Current angle of the machine gun */
  int         iGunAngle;
  
  /** Pointer to machine gun graphics */
  BITMAP*     iGunGraphics;
  
  /** Machine gun reloading delay. */
  int         iGunDelay;
  
  /** Moving velocity of this car. */
  float       iVelocity;
  
  /** Relative position of the gun nose when rotation angle is 0 */
  eng2d::Vec2D  iGunNosePos;


private:
  ///
  /// Private interface
  /// =================

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: playercar.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-13 14:29:24+02  lamminsa
 * Constructor takes the route parameter.
 *
 */
 
